#!/usr/bin/env node

const fs = require("fs");
const path = require("path");
const { execSync } = require("child_process");
const archiver = require("archiver");

function getCommitHash() {
	try {
		return execSync("git rev-parse --short HEAD", { encoding: "utf8" }).trim();
	} catch (error) {
		console.warn("Warning: Could not get git commit hash:", error.message);
		return "unknown";
	}
}

function getCurrentDate() {
	const now = new Date();
	const year = now.getFullYear();
	const month = String(now.getMonth() + 1).padStart(2, "0");
	const day = String(now.getDate()).padStart(2, "0");
	return `${year}-${month}-${day}`;
}

function generateSourceZip() {
	const commitHash = getCommitHash();
	const currentDate = getCurrentDate();
	const zipFileName = `oekohaus-roessing-quellcode.zip`;
	const zipPath = path.join(__dirname, "..", "static", zipFileName);

	// Ensure static directory exists
	const staticDir = path.dirname(zipPath);
	if (!fs.existsSync(staticDir)) {
		fs.mkdirSync(staticDir, { recursive: true });
	}

	return new Promise((resolve, reject) => {
		const output = fs.createWriteStream(zipPath);
		const archive = archiver("zip", {
			zlib: { level: 9 },
		});

		output.on("close", () => {
			console.log(
				`Generated source zip: ${zipFileName} (${archive.pointer()} bytes)`,
			);
			resolve({ fileName: zipFileName, filePath: zipPath });
		});

		archive.on("error", (err) => {
			reject(err);
		});

		archive.pipe(output);

		// Add metadata file with build info
		const metadata = `# Build Information
project: "Ökohaus Rössing"
description: "Kompletter Quellcode der Ökohaus Rössing Website"
generated: "${currentDate}"
commit: "${commitHash}"
license: "MIT"
`;
		archive.append(metadata, { name: "build-info.yml" });

		// Add all files except build artifacts and dependencies
		archive.glob("**/*", {
			cwd: path.join(__dirname, ".."),
			ignore: [
				"node_modules/**",
				"build/**",
				".git/**",
				"static/oekohaus-roessing-quellcode.zip",
				"*.log",
				".env*",
				"coverage/**",
				".nyc_output/**",
				"dist/**",
				"tmp/**",
				".cache/**",
			],
		});

		archive.finalize();
	});
}

if (require.main === module) {
	generateSourceZip()
		.then(({ fileName }) => {
			console.log(`Source ZIP generated successfully: ${fileName}`);
		})
		.catch((error) => {
			console.error("Error generating source ZIP:", error);
			process.exit(1);
		});
}

module.exports = { generateSourceZip };
